import { Effect, Entity, GameMode, ItemStack, Player, TicksPerSecond } from "@minecraft/server";
import { entityManager } from "../class/entityClass";
import { DEBUG } from "../config";


/**
 * The main function for the sounds
 * @param {soundId} sound - The ID of the sound being played
 * @param {Player} player - The player(s) that the sound will play
 * @param {Number} param0 - The Volume and Pitch for the sound
 */
export function runSoundAction(sound, player, {volume = 1, pitch = 1}) {
    if (!sound || sound == " ") {
        console.warn("No valid sound id was provided!");
        return;
    }
    const dim = player.dimension;
    const loc = player.location;
    dim.playSound(sound, loc, {
        volume: volume,
        pitch: pitch
    })
}

/**
 * The main function for adding tags
 * @param {tags} tag - The Tag
 * @param {Player} player - The Player
 * @param {Player} selector - The selected player if provide
 */
export function runAddTag(tag, player, selector) {
    if (!tag || tag == " ") {
        console.warn("No valid tag was provided!");
        return;
    }
    if (selector == undefined) {
        const checkTag = player.hasTag(tag);
        try {
            if (checkTag) {
                player.sendMessage(`§cPlayer ${player.nameTag} already has the tag (${tag}) or has too many tags.`);
                return;
            } else if (!checkTag) {
                player.addTag(tag);
                player.sendMessage(`Successfully gave ${player.nameTag} the (${tag}) tag`);
                return;
            }
        } catch (e) {
            console.error(`Error has occur!\n`, e);
        }
    } else if (selector != undefined) {
        const checkTag = selector.hasTag(tag);
        try {
            if (checkTag) {
                player.sendMessage(`§cPlayer ${selector.nameTag} already has the tag (${tag}) or has too many tags.`);
                return;
            } else if (!checkTag) {
                selector.addTag(tag);
                player.sendMessage(`Successfully gave ${selector.nameTag} the (${tag}) tag`);
                return;
            }
        } catch (e) {
            console.error(`Error has occur!\n`, e);
        }
    }
}

export function runRemoveTag(tag, player, selector) {
    if (!tag || tag == " ") {
        console.warn("No valid tag was provided!");
        return;
    }
    if (selector == undefined) {
        const checkTag = player.hasTag(tag);
        try {
            if (checkTag) {
                player.removeTag(tag);
                player.sendMessage(`Successfully removed (${tag}) tag from ${player.nameTag}`);
                return;
            } else if (!checkTag) {
                player.sendMessage(`§cPlayer ${player.nameTag} doesn't have the (${tag}) tag.`);
                return;
            }
        } catch (e) {
            console.error(`Error has occur!\n`, e);
        }
    } else if (selector != undefined) {
        const checkTag = selector.hasTag(tag);
        try {
            if (checkTag) {
                selector.removeTag(tag);
                player.sendMessage(`Successfully removed (${tag}) tag from ${selector.nameTag}`);
                return;
            } else if (!checkTag) {
                selector.sendMessage(`§cPlayer ${selector.nameTag} doesn't have the (${tag}) tag.`);
                return;
            }
        } catch (e) {
            console.error(`Error has occur!\n`, e);
        }
    }
}

/**
 * 
 * @param {Effect} effect 
 * @param {Player} player 
 * @param {Player} selector 
 * @param {number|boolean} param3 
 */
export function runGiveEffect(effect, player, selector, {amplifier = 0, duration = 1, particles = true}) {
    if (!effect || effect == " ") {
        console.warn("No valid Effect was provided!");
        return;
    }
    const effectID  = effect.toLowerCase().replace(/\s+/g, "_");
    duration = duration * TicksPerSecond;
    if (selector == undefined) {
        if (DEBUG) console.log(`Effect Given!\nPlayer: ${player.nameTag}\nEffect: ${effectID} as ${effect}\nAmplifier: ${amplifier}\nDuration: ${duration}\nShow Particles: ${particles}`);
        player.addEffect(effectID, duration, {
            amplifier: amplifier,
            showParticles: particles
        });
    } else if (selector != undefined) {
        selector.addEffect(effectID, duration, {
            amplifier: amplifier,
            showParticles: particles
        });
    }
}

/**
 * 
 * @param {GameMode} gamemode - The GameMode
 * @param {Player} player - The player
 * @param {Player} selector - The selected player
 */
export function runGamemode(gamemode, player, selector) {
    if (!gamemode || gamemode == " ") {
        console.warn("No valid GameMode was provided!");
        return;
    }
    if (DEBUG) console.log(`Gamemode: ${gamemode}\nPlayer: ${player.nameTag}`);
    if (selector == undefined) {
        player.setGameMode(gamemode);
        if (DEBUG) console.log(`Changed Player ${player.nameTag}'s gamemode to ${gamemode}`);
    } else if (selector != undefined) {
        if (DEBUG) console.log(`Selector: ${selector.nameTag}`);
        selector.setGameMode(gamemode);
    }
}

/**
 * The main function for giving items.
 * @param {ItemStack} item - Item that is being given.
 * @param {Player} player - The player(s) that is getting the item.
 * @param {Player} selector - The selected player(s) that is getting the item.
 */
export function runGiveItem(item, player, selector) {
    if (!item || item == " ") {
        console.warn("No valid Item was provided!");
        return;
    }
    if (selector == undefined) {
        const inventory = player.getComponent("inventory");
        if (inventory.container.emptySlotsCount != 0) {
            // If there is room in the inventory, add it to the inventory
            inventory.container.addItem(item);
            return;
        } else if (inventory.container.emptySlotsCount == 0) {
            // Else, spawn it in the world at their location
            const loc = player.location;
            entityManager.spawnItemAnywhere(item, loc, player.dimension);
        }
    } else if (selector != undefined) {
        const inventory = selector.getComponent("inventory");
        if (inventory.container.emptySlotsCount != 0) {
            // If there is room in the inventory, add it to the inventory
            inventory.container.addItem(item);
            return;
        } else if (inventory.container.emptySlotsCount == 0) {
            // Else, spawn it in the world at their location
            const loc = selector.location;
            entityManager.spawnItemAnywhere(item, loc, selector.dimension);
        }
    }
}
/**
 * The main function for teleporting
 * @param {import("@minecraft/server").Vector3} location - The location being teleported
 * @param {Player} player - The Player
 * @param {Player} selector - The selected Player
 */
export function runTeleportAction(location, player, selector, dimension) {
    if (!location || location == " ") {
        console.warn("No valid Location was provided!");
        return;
    }
    if (selector == undefined) {
        player.teleport(location, {
            dimension: dimension
        });
    } else if (selector != undefined) {
        selector.teleport(location, {
            dimension: dimension
        });
    }
}

/**
 * The main function for spawning a mob on a player
 * @param {import("@minecraft/server").Vector3} location - The location that the entity will spawn.
 * @param {Entity} entity - The entity that is being spawned.
 * @param {Player} player - The player that submitted the form. Is the target if selector is undefined.
 * @param {Player} selector - The selected player that is being effected.
 */
export function runSpawnEntity(location, entity, player, selector) {
    if (selector == undefined) {
        player.dimension.spawnEntity(entity, location)
    } else if (selector != undefined) {
        selector.dimension.spawnEntity(entity, location)
    }
}